<?php

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Generate the authentication header for Finmo API.
 *
 * @param string $access_key Finmo access key.
 * @param string $secret_key Finmo secret key.
 *
 * @return string Authentication header.
 */
function finmo_generate_auth_header($access_key, $secret_key)
{
    if (empty($access_key) || empty($secret_key)) {
        finmo_log('[Auth] Missing access or secret key for auth header.', 'error');
        return '';
    }

    return "Basic " . base64_encode("{$access_key}:{$secret_key}");
}

/**
 * Make an API request using wp_remote_post.
 *
 * @param string $url          The URL for the API request.
 * @param string $access_key   Finmo access key.
 * @param string $secret_key   Finmo secret key.
 * @param string $current_env  Current environment.
 * @param array  $args         API arguments.
 *
 * @return mixed|null Decoded JSON response or null on failure.
 */
function finmo_api_request($url, $access_key, $secret_key, $current_env, $args)
{
    try {
        $authentication = finmo_generate_auth_header($access_key, $secret_key);
        $body           = json_encode($args);

        finmo_log("[API Request] URL: {$url} | Env: {$current_env} | Body: {$body}", 'debug');

        $response = wp_remote_post(
            esc_url_raw($url),
            [
                'method'      => 'POST',
                'timeout'     => 45,
                'redirection' => 5,
                'httpversion' => '1.0',
                'blocking'    => true,
                'headers'     => [
                    'Authorization'   => $authentication,
                    'Content-Type'    => 'application/json',
                    'x-env'           => $current_env,
                    'x-plugin-type'   => 'woocommerce',
                ],
                'body'        => $body,
            ]
        );

        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            finmo_log("[API Error] {$error_message}", 'error');
            return null;
        }

        $response_body = wp_remote_retrieve_body($response);
        $decoded       = json_decode($response_body);

        if (json_last_error() !== JSON_ERROR_NONE) {
            finmo_log("[API Error] Invalid JSON response: {$response_body}", 'error');
            return null;
        }

        return $decoded;
    } catch (Exception $e) {
        finmo_log('[API Exception] ' . $e->getMessage(), 'error');
        return null;
    }
}

/**
 * Validate API key with Finmo CP.
 *
 * @param array $args API arguments.
 *
 * @return mixed API response object|null
 */
function finmo_validate_api_key($args)
{
    $api_url = FINMO_CP_BASE_URL . "/plugin-api-key/validate";
    $body    = json_encode($args);

    try {
        $response = wp_remote_post(
            esc_url_raw($api_url),
            [
                'method'      => 'POST',
                'timeout'     => 45,
                'redirection' => 5,
                'httpversion' => '1.0',
                'blocking'    => true,
                'headers'     => [
                    'Content-Type' => 'application/json',
                ],
                'body'        => $body,
            ]
        );

        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            finmo_log("[API Key Validation] Failed: {$error_message}", 'error');
            return null;
        }

        $decoded = json_decode(wp_remote_retrieve_body($response));
        if (json_last_error() !== JSON_ERROR_NONE) {
            finmo_log("[API Key Validation] Invalid JSON response", 'error');
            return null;
        }

        return $decoded;
    } catch (Exception $e) {
        finmo_log('[API Key Validation Exception] ' . $e->getMessage(), 'error');
        return null;
    }
}

/**
 * Create checkout request.
 */
function finmo_create_checkout_request($access_key, $secret_key, $current_env, $args, $order_id)
{
    $api_url       = FINMO_API_BASE_URL . "/v1/plugin/woocommerce/checkout";
    $response_data = finmo_api_request($api_url, $access_key, $secret_key, $current_env, $args);

    if (!$response_data || empty($response_data->success)) {
        finmo_log("[Checkout API] Failed for Order {$order_id} | Response: " . json_encode($response_data), 'error');
        return $response_data;
    }

    finmo_log("[Checkout API] Success for Order {$order_id} | Checkout ID: {$response_data->data->checkout_id}", 'info');
    return $response_data;
}

/**
 * Create refund request.
 */
function finmo_create_refund_request($access_key, $secret_key, $current_env, $args, $order_id)
{
    $api_url       = FINMO_API_BASE_URL . "/v1/plugin/woocommerce/refund";
    $response_data = finmo_api_request($api_url, $access_key, $secret_key, $current_env, $args);

    if (!$response_data || empty($response_data->success)) {
        finmo_log("[Refund API] Failed for Order {$order_id} | Response: " . json_encode($response_data), 'error');
        return $response_data;
    }

    finmo_log("[Refund API] Success for Order {$order_id} | Refund ID: {$response_data->data->refund_id}", 'info');
    return $response_data;
}